define(['angular', 'app', 'userService', 'localResourceDirectoryService', 'formatter', 'NotificationsConstants'], function (angular, app) {
	"use strict";
	app.service('NotificationsPreferencesService', function ($http, $filter, $q, localResourceDirectoryService, userService, formatter, NotificationsConstants) {

		var NotificationsService = {},
			notificationTransforms = [],
			isPreferenceStored = null,
			preferencesStoragePromise = $q.defer();

		NotificationsService.initializedPromise = null;
		NotificationsService.preferencesStoragePromise = preferencesStoragePromise.promise;

		NotificationsService.dataPointers = {
			notificationPreference: null,
			unreadNotificationsCount: 0,
			emailAddress: null,
			timeZonePreference: null,
			patientPreference: null
		};

		NotificationsService.initializeResources = function () {

			if (NotificationsService.initializedPromise !== null) {
				return NotificationsService.initializedPromise.promise;
			}

			NotificationsService.initializedPromise = $q.defer();

			var NotificationPreferencesInitializationPromise = $q.defer();

			userService.getUserData().then(function (user) {
				NotificationsService.dataPointers.notificationPreference = {
					"firstName": user.firstName,
					"lastName": user.lastName,
					"dateOfBirth": user.patient.dateOfBirth,
					"optInSelected": null
				};
				NotificationPreferencesInitializationPromise.resolve($q.when("Default Notification Preferences Initialized"));
			});

			$q.all({notificationPreferencesInitializationPromise: NotificationPreferencesInitializationPromise, resources: localResourceDirectoryService.fetch()})
				.then(function (responses) {
					NotificationsService.initializedPromise.resolve(responses.resources);
				});

			return NotificationsService.initializedPromise.promise;
		};

		var makeNotificationRequest = function (config, linkTitle) {
			var defer = $q.defer();

			function sendRequest(config) {
				var notificationTransformKeys = Object.keys(notificationTransforms);

				function transformEntry(item) {
					item.formattedEffectiveDisplayDate = formatter.getFormattedFrontendDateTime(item.effectiveDisplayDate);

					notificationTransforms.forEach(function (transformFn) {
						item = transformFn(item);
					});

					return item;
				}

				if(config.data) {
					delete config.data.link
				}
				
				$http(config).success(function (response, status) {
					if (status !== 204) {
						switch (response['objectType']) {
							case "Notification" :
								response = transformEntry(response);
								break;
							case "Notifications" :
								if (response.notifications) {
									response.notifications.forEach(function (item) {
										item = transformEntry(item);
									});
								} else {
									response.notifications = [];
								}
								break;
						}
						preferencesStoragePromise.resolve();
						isPreferenceStored = true;
						defer.resolve(response);
					} else {
						isPreferenceStored = false;
						defer.reject(response);
					}
				}).error(function (error) {
					defer.reject(error);
				});
			}

			if (config.url) {
				sendRequest(config);
			} else {
				NotificationsService.initializeResources().then(
					function (resources) {
						config.url = resources[linkTitle];
						sendRequest(config);
					});
			}

			return defer.promise;
		};

		NotificationsService.createNotification = function (item) {
			var requestPromise = makeNotificationRequest({method: "POST", data: item}, "notification");
			requestPromise.then(function (response) {
				return response;
			});
			return requestPromise;
		};

		NotificationsService.getNotificationsPreference = function () {
			var requestPromise = makeNotificationRequest({method: "GET"}, "notification-preference");
			requestPromise.then(function (response) {
				NotificationsService.dataPointers.notificationPreference = response;
			});
			return requestPromise;
		};

		NotificationsService.createNotificationsPreference = function (optInSelected) {
			isPreferenceStored = true;
			var defer = $q.defer();
			NotificationsService.initializeResources().then(function () {
				var updatedPreference = angular.copy(NotificationsService.dataPointers.notificationPreference) || {};
				updatedPreference.optInSelected = optInSelected;
				var requestPromise = makeNotificationRequest({method: "POST", data: updatedPreference}, "notification-preference");
				requestPromise.then(function (response) {
					NotificationsService.dataPointers.notificationPreference = response;
					defer.resolve();
				});
			});
			return defer.promise;
		};

		NotificationsService.updateNotificationsPreference = function (optInSelected) {
			var updatedPreference = angular.copy(NotificationsService.dataPointers.notificationPreference) || {};
			updatedPreference.optInSelected = optInSelected;
			var requestPromise = makeNotificationRequest({method: "PUT", data: updatedPreference}, "notification-preference");
			requestPromise.then(function (response) {
				NotificationsService.dataPointers.notificationPreference = response;
			});
			return requestPromise;
		};

		NotificationsService.getNotificationsPreferenceAndCount = function () {
			var notificationPreference = angular.copy(NotificationsService.dataPointers.notificationPreference) || {};
            var notificationType = notificationPreference.optInSelected ? "FIT_HEART,STAFF" : "FIT_HEART";

			var requestPromise = makeNotificationRequest({
                method: "GET",
                params: {
                    type: notificationType
                }
            }, "notification-preference-and-count");

			requestPromise.then(function (response) {
				NotificationsService.dataPointers.notificationPreference = response.notificationPreference;
				NotificationsService.dataPointers.unreadNotificationsCount = response.notificationsCount;
			});
			return requestPromise;
		};

		NotificationsService.isPreferenceStored = function () {
			return isPreferenceStored;
		};

		NotificationsService.getPreferencesData = function () {
			var requestPromise = makeNotificationRequest({method: "GET"}, "preferences");
			requestPromise.then(function (response) {
				NotificationsService.dataPointers.emailAddress = response.emailAddress;
				NotificationsService.dataPointers.timeZonePreference = response.timeZone;
				NotificationsService.dataPointers.patientPreference = response;
			});
			return requestPromise;
		};

		NotificationsService.createPreferencesData = function (emailAddressPreference, timeZonePreference) {
			var defer = $q.defer();
			var requestPromise;

			var updatedPreference = angular.copy(NotificationsService.dataPointers.patientPreference) || {};

			NotificationsService.initializeResources().then(function () {

				angular.element.extend(true, updatedPreference, {emailAddress: emailAddressPreference || null, timeZone: timeZonePreference});

				requestPromise = makeNotificationRequest({method: "POST", data: updatedPreference}, "preferences");

				requestPromise.then(function (response) {
					NotificationsService.dataPointers.emailAddress = response.emailAddress;
					NotificationsService.dataPointers.timeZonePreference = response.timeZone;
					NotificationsService.dataPointers.patientPreference = response;
					defer.resolve();
				});
			});
			return defer.promise;
		};

		NotificationsService.updatePreferencesData = function (emailAddressPreference, timeZonePreference) {
			var requestPromise;

			var updatedPreference = angular.copy(NotificationsService.dataPointers.patientPreference) || {};


			angular.element.extend(true, updatedPreference, {emailAddress: emailAddressPreference || null, timeZone: timeZonePreference});
			
			requestPromise = makeNotificationRequest({method: "PUT", data: updatedPreference}, "preferences");

			requestPromise.then(function (response) {
				NotificationsService.dataPointers.emailAddress = response.emailAddress;
				NotificationsService.dataPointers.timeZonePreference = response.timeZone;
				NotificationsService.dataPointers.patientPreference = response;
			});
			return requestPromise;
		};

		return NotificationsService;
	});
});